const mysql = require('mysql2/promise');
const bcrypt = require('bcrypt');
const { v4: uuidv4 } = require('uuid');

// Sample data
const users = [
  {
    id: uuidv4(),
    name: 'Admin User',
    email: 'adminx@school.com',
    password: 'admin123',
    role: 'admin'
  },
  {
    id: uuidv4(),
    name: 'Teacher User',
    email: 'teacher@school.com',
    password: 'teacher123',
    role: 'teacher'
  },
  {
    id: uuidv4(),
    name: 'John Doe',
    email: 'john.doe@school.com',
    password: '123456',
    role: 'student'
  },
  {
    id: uuidv4(),
    name: 'Jane Smith',
    email: 'jane.smith@school.com',
    password: '123456',
    role: 'student'
  },
  {
    id: uuidv4(),
    name: 'Parent User',
    email: 'parent@school.com',
    password: 'parent123',
    role: 'parent'
  }
];

const students = [
  {
    id: uuidv4(),
    user_id: null, // Will be set after users are inserted
    student_id: 'S10001',
    name: 'John Doe',
    grade_level: '10',
    class_section: 'A',
    date_of_birth: '2005-05-15',
    gender: 'male',
    address: '123 School St, City',
    phone: '555-123-4567',
    email: 'john.doe@school.com',
    parent_name: 'Parent Doe',
    parent_phone: '555-987-6543',
    parent_email: 'parent@school.com',
    enrollment_date: '2022-09-01',
    profile_image: 'https://randomuser.me/api/portraits/men/1.jpg'
  },
  {
    id: uuidv4(),
    user_id: null, // Will be set after users are inserted
    student_id: 'S10002',
    name: 'Jane Smith',
    grade_level: '10',
    class_section: 'A',
    date_of_birth: '2005-08-22',
    gender: 'female',
    address: '456 Academy Ave, City',
    phone: '555-234-5678',
    email: 'jane.smith@school.com',
    parent_name: 'Parent Smith',
    parent_phone: '555-876-5432',
    parent_email: 'parent.smith@email.com',
    enrollment_date: '2022-09-01',
    profile_image: 'https://randomuser.me/api/portraits/women/1.jpg'
  }
];

const courses = [
  {
    id: uuidv4(),
    course_code: 'MATH101',
    name: 'Mathematics',
    description: 'Basic mathematics course covering algebra and geometry',
    credit_hours: 4,
    teacher_id: null, // Will be set after users are inserted
    grade_level: '10',
    class_section: 'A',
    academic_year: '2023-2024',
    semester: 'Fall'
  },
  {
    id: uuidv4(),
    course_code: 'ENG101',
    name: 'English',
    description: 'English language and literature',
    credit_hours: 3,
    teacher_id: null, // Will be set after users are inserted
    grade_level: '10',
    class_section: 'A',
    academic_year: '2023-2024',
    semester: 'Fall'
  },
  {
    id: uuidv4(),
    course_code: 'SCI101',
    name: 'Science',
    description: 'Introduction to scientific principles',
    credit_hours: 4,
    teacher_id: null, // Will be set after users are inserted
    grade_level: '10',
    class_section: 'A',
    academic_year: '2023-2024',
    semester: 'Fall'
  },
  {
    id: uuidv4(),
    course_code: 'HIST101',
    name: 'History',
    description: 'World history and civilization',
    credit_hours: 3,
    teacher_id: null, // Will be set after users are inserted
    grade_level: '10',
    class_section: 'A',
    academic_year: '2023-2024',
    semester: 'Fall'
  }
];

async function seedDatabase() {
  // Create connection
  const connection = await mysql.createConnection({
    host: '38.46.219.162',
    user: 'drimamskom',
    password: 'D15n4kj4t1m', // Replace with actual password in production
    port: 30012,
    database: 'school_management',
    multipleStatements: true
  });

  try {
    console.log('Connected to database. Starting seed process...');

    // Insert users with hashed passwords
    for (const user of users) {
      const salt = await bcrypt.genSalt(10);
      const hashedPassword = await bcrypt.hash(user.password, salt);
      
      await connection.execute(
        'INSERT INTO users (id, name, email, password, role) VALUES (?, ?, ?, ?, ?)',
        [user.id, user.name, user.email, hashedPassword, user.role]
      );
      console.log(`User ${user.name} inserted`);
    }

    // Get user IDs for linking
    const [userRows] = await connection.execute('SELECT id, email, role FROM users');
    const userMap = {};
    userRows.forEach(user => {
      userMap[user.email] = user.id;
    });

    // Update student user_ids
    students.forEach(student => {
      student.user_id = userMap[student.email];
    });

    // Insert students
    for (const student of students) {
      await connection.execute(
        `INSERT INTO students 
        (id, user_id, student_id, name, grade_level, class_section, date_of_birth, gender, 
        address, phone, email, parent_name, parent_phone, parent_email, enrollment_date, profile_image) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          student.id, student.user_id, student.student_id, student.name, student.grade_level, 
          student.class_section, student.date_of_birth, student.gender, student.address, 
          student.phone, student.email, student.parent_name, student.parent_phone, 
          student.parent_email, student.enrollment_date, student.profile_image
        ]
      );
      console.log(`Student ${student.name} inserted`);
    }

    // Get teacher ID
    const teacherId = userMap['teacher@school.com'];
    
    // Update course teacher_ids
    courses.forEach(course => {
      course.teacher_id = teacherId;
    });

    // Insert courses
    for (const course of courses) {
      await connection.execute(
        `INSERT INTO courses 
        (id, course_code, name, description, credit_hours, teacher_id, grade_level, 
        class_section, academic_year, semester) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          course.id, course.course_code, course.name, course.description, course.credit_hours,
          course.teacher_id, course.grade_level, course.class_section, course.academic_year, course.semester
        ]
      );
      console.log(`Course ${course.name} inserted`);
    }

    // Enroll students in courses
    const [studentRows] = await connection.execute('SELECT id FROM students');
    const [courseRows] = await connection.execute('SELECT id FROM courses');
    
    for (const student of studentRows) {
      for (const course of courseRows) {
        await connection.execute(
          `INSERT INTO student_courses (id, student_id, course_id, enrollment_date, status)
          VALUES (?, ?, ?, ?, ?)`,
          [uuidv4(), student.id, course.id, '2023-09-01', 'active']
        );
      }
      console.log(`Enrolled student ${student.id} in all courses`);
    }

    // Insert assignments
    const assignments = [
      {
        id: uuidv4(),
        course_id: courseRows[0].id, // Math course
        title: 'Algebra Quiz',
        description: 'Quiz covering basic algebraic equations',
        due_date: '2023-10-15',
        total_points: 100,
        assignment_type: 'quiz',
        status: 'upcoming'
      },
      {
        id: uuidv4(),
        course_id: courseRows[1].id, // English course
        title: 'Essay Assignment',
        description: 'Write a 500-word essay on a topic of your choice',
        due_date: '2023-10-20',
        total_points: 100,
        assignment_type: 'homework',
        status: 'upcoming'
      },
      {
        id: uuidv4(),
        course_id: courseRows[2].id, // Science course
        title: 'Lab Report',
        description: 'Complete the lab report for the experiment conducted in class',
        due_date: '2023-10-25',
        total_points: 100,
        assignment_type: 'project',
        status: 'upcoming'
      }
    ];

    for (const assignment of assignments) {
      await connection.execute(
        `INSERT INTO assignments 
        (id, course_id, title, description, due_date, total_points, assignment_type, status) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          assignment.id, assignment.course_id, assignment.title, assignment.description,
          assignment.due_date, assignment.total_points, assignment.assignment_type, assignment.status
        ]
      );
      console.log(`Assignment ${assignment.title} inserted`);
    }

    // Insert grades
    for (const student of studentRows) {
      for (const course of courseRows) {
        await connection.execute(
          `INSERT INTO grades 
          (id, student_id, course_id, score, grade_letter, grading_period, date_graded) 
          VALUES (?, ?, ?, ?, ?, ?, ?)`,
          [
            uuidv4(), student.id, course.id, 
            Math.floor(Math.random() * 30) + 70, // Random score between 70-100
            'B', 'Q1', '2023-10-01'
          ]
        );
      }
      console.log(`Grades inserted for student ${student.id}`);
    }

    // Insert attendance records
    const attendanceStatuses = ['present', 'present', 'present', 'present', 'absent', 'late'];
    const dates = ['2023-09-05', '2023-09-12', '2023-09-19', '2023-09-26', '2023-10-03', '2023-10-10'];
    
    for (const student of studentRows) {
      for (const course of courseRows) {
        for (const date of dates) {
          const status = attendanceStatuses[Math.floor(Math.random() * attendanceStatuses.length)];
          await connection.execute(
            `INSERT INTO attendance 
            (id, student_id, course_id, date, status) 
            VALUES (?, ?, ?, ?, ?)`,
            [uuidv4(), student.id, course.id, date, status]
          );
        }
      }
      console.log(`Attendance records inserted for student ${student.id}`);
    }

    // Insert schedule
    const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday'];
    const times = [
      { start: '08:00:00', end: '09:30:00' },
      { start: '10:00:00', end: '11:30:00' },
      { start: '12:30:00', end: '14:00:00' },
      { start: '14:30:00', end: '16:00:00' }
    ];
    
    for (let i = 0; i < courseRows.length; i++) {
      await connection.execute(
        `INSERT INTO schedule 
        (id, course_id, day_of_week, start_time, end_time, room) 
        VALUES (?, ?, ?, ?, ?, ?)`,
        [
          uuidv4(), courseRows[i].id, days[i % days.length], 
          times[i % times.length].start, times[i % times.length].end,
          `Room ${101 + i}`
        ]
      );
      console.log(`Schedule inserted for course ${courseRows[i].id}`);
    }

    // Insert fees
    const feeTypes = ['tuition', 'exam', 'transportation', 'library'];
    const amounts = [5000, 500, 1000, 200];
    
    for (const student of studentRows) {
      for (let i = 0; i < feeTypes.length; i++) {
        await connection.execute(
          `INSERT INTO fees 
          (id, student_id, fee_type, amount, due_date, payment_status) 
          VALUES (?, ?, ?, ?, ?, ?)`,
          [
            uuidv4(), student.id, feeTypes[i], amounts[i],
            '2023-10-31', i === 0 ? 'paid' : 'unpaid'
          ]
        );
      }
      console.log(`Fees inserted for student ${student.id}`);
    }

    // Insert announcements
    const announcements = [
      {
        id: uuidv4(),
        title: 'School Holiday',
        content: 'School will be closed on October 31st for the holiday.',
        author_id: userMap['admin@school.com'],
        target_audience: 'all',
        publish_date: '2023-10-15 10:00:00',
        is_pinned: true
      },
      {
        id: uuidv4(),
        title: 'Parent-Teacher Meeting',
        content: 'Parent-teacher meetings will be held on November 5th. Please check your email for the schedule.',
        author_id: userMap['admin@school.com'],
        target_audience: 'parents',
        publish_date: '2023-10-20 14:00:00',
        is_pinned: false
      },
      {
        id: uuidv4(),
        title: 'Exam Schedule',
        content: 'Final exams will begin on December 10th. The detailed schedule will be posted next week.',
        author_id: userMap['teacher@school.com'],
        target_audience: 'students',
        publish_date: '2023-10-25 09:00:00',
        is_pinned: true
      }
    ];
    
    for (const announcement of announcements) {
      await connection.execute(
        `INSERT INTO announcements 
        (id, title, content, author_id, target_audience, publish_date, is_pinned) 
        VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          announcement.id, announcement.title, announcement.content, announcement.author_id,
          announcement.target_audience, announcement.publish_date, announcement.is_pinned
        ]
      );
      console.log(`Announcement "${announcement.title}" inserted`);
    }

    // Insert messages
    const messages = [
      {
        id: uuidv4(),
        sender_id: userMap['teacher@school.com'],
        receiver_id: userMap['john.doe@school.com'],
        content: 'Please submit your assignment by tomorrow.',
        timestamp: '2023-10-10 13:45:00',
        read: true
      },
      {
        id: uuidv4(),
        sender_id: userMap['john.doe@school.com'],
        receiver_id: userMap['teacher@school.com'],
        content: 'I will submit it today. Thank you.',
        timestamp: '2023-10-10 14:30:00',
        read: true
      },
      {
        id: uuidv4(),
        sender_id: userMap['admin@school.com'],
        receiver_id: userMap['john.doe@school.com'],
        content: 'Please visit the admin office to collect your ID card.',
        timestamp: '2023-10-12 10:15:00',
        read: false
      }
    ];
    
    for (const message of messages) {
      await connection.execute(
        `INSERT INTO messages 
        (id, sender_id, receiver_id, content, timestamp, read) 
        VALUES (?, ?, ?, ?, ?, ?)`,
        [
          message.id, message.sender_id, message.receiver_id,
          message.content, message.timestamp, message.read
        ]
      );
      console.log(`Message from ${message.sender_id} to ${message.receiver_id} inserted`);
    }

    // Insert documents
    const documents = [
      {
        id: uuidv4(),
        user_id: userMap['john.doe@school.com'],
        title: 'Student ID Card',
        description: 'Digital copy of student ID card',
        file_path: '/documents/id_cards/john_doe_id.pdf',
        file_type: 'application/pdf',
        file_size: 256000,
        document_type: 'id_card',
        status: 'active'
      },
      {
        id: uuidv4(),
        user_id: userMap['john.doe@school.com'],
        title: 'Report Card - Q1',
        description: 'First quarter report card',
        file_path: '/documents/report_cards/john_doe_q1.pdf',
        file_type: 'application/pdf',
        file_size: 512000,
        document_type: 'report_card',
        status: 'active'
      },
      {
        id: uuidv4(),
        user_id: userMap['jane.smith@school.com'],
        title: 'Student ID Card',
        description: 'Digital copy of student ID card',
        file_path: '/documents/id_cards/jane_smith_id.pdf',
        file_type: 'application/pdf',
        file_size: 256000,
        document_type: 'id_card',
        status: 'active'
      }
    ];
    
    for (const document of documents) {
      await connection.execute(
        `INSERT INTO documents 
        (id, user_id, title, description, file_path, file_type, file_size, document_type, status) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          document.id, document.user_id, document.title, document.description,
          document.file_path, document.file_type, document.file_size,
          document.document_type, document.status
        ]
      );
      console.log(`Document "${document.title}" inserted`);
    }

    // Insert payment methods
    const paymentMethods = [
      {
        id: uuidv4(),
        user_id: userMap['john.doe@school.com'],
        method_type: 'credit_card',
        card_number: 'XXXX-XXXX-XXXX-1234',
        card_holder: 'John Doe',
        expiry_date: '12/25',
        is_default: true
      },
      {
        id: uuidv4(),
        user_id: userMap['john.doe@school.com'],
        method_type: 'bank_transfer',
        bank_name: 'National Bank',
        account_number: 'XXXXXXXXXXXX5678',
        account_holder: 'John Doe',
        is_default: false
      },
      {
        id: uuidv4(),
        user_id: userMap['jane.smith@school.com'],
        method_type: 'debit_card',
        card_number: 'XXXX-XXXX-XXXX-5678',
        card_holder: 'Jane Smith',
        expiry_date: '10/24',
        is_default: true
      }
    ];
    
    for (const method of paymentMethods) {
      await connection.execute(
        `INSERT INTO payment_methods 
        (id, user_id, method_type, card_number, card_holder, expiry_date, bank_name, account_number, account_holder, is_default) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          method.id, method.user_id, method.method_type, 
          method.card_number || null, method.card_holder || null, method.expiry_date || null,
          method.bank_name || null, method.account_number || null, method.account_holder || null,
          method.is_default
        ]
      );
      console.log(`Payment method for ${method.user_id} inserted`);
    }

    console.log('Database seeding completed successfully!');
  } catch (error) {
    console.error('Error seeding database:', error);
  } finally {
    await connection.end();
    console.log('Database connection closed');
  }
}

seedDatabase();